<?php

/**
 * Class settings
 *
 * Manages plugin settings, including retrieval, updating, and caching.
 *
 * This class provides methods for getting and saving settings, as well as clearing the cache.
 * It interacts with the WordPress database to store and retrieve settings.
 */
class settings {

	/**
	 * Get settings
	 *
	 * @return array
	 */
	public function get_settings() {
		global $wpdb;
		
		$sSettings = array();
		$sSettings['settings']['mysql_error'] = true;
		
		if(empty($wpdb)) {
			return;
		}
		
		$sQuery = "
		SELECT * FROM 
			". $wpdb->prefix ."ovis
		WHERE 
			OVIS_ID = 1
		";
		
		/**
		 * Retrieve & store results
		 */
		$sResult = $wpdb->get_results($sQuery , ARRAY_A);
		
		/**
		 * Check for existing results
		 */
		if(!empty($sResult[0])) {

			/**
			 * Check for valid results
			 */
			if(is_array($sResult[0])) {

				$sResult = $sResult[0];
				
				/**
				 * Remove MySQL error
				 */
				unset($sSettings['settings']['mysql_error']);
				$sSettings['settings']['authentication_key'] 				= $sResult['OVIS_authentication'];
				$sSettings['settings']['authentication_header'] 			= array('Content-Type: application/json; charset=utf-8','Authentication: '.$sResult['OVIS_authentication'],'Version: ' . OVIS_VERSION);
				$sSettings['settings']['button_bg_color'] 					= isset($sResult['OVIS_button_bg_color']) ? $sResult['OVIS_button_bg_color'] : '#DFDFDF';
				$sSettings['settings']['button_text_color'] 				= isset($sResult['OVIS_button_text_color']) ? $sResult['OVIS_button_text_color'] : '#333333';
				$sSettings['settings']['object_bg_color'] 					= isset($sResult['OVIS_object_bg_color']) ? $sResult['OVIS_object_bg_color'] : '#DFDFDF';
				$sSettings['settings']['object_text_color'] 				= isset($sResult['OVIS_object_text_color']) ? $sResult['OVIS_object_text_color'] : '#333333';
				$sSettings['settings']['items_per_page'] 					= isset($sResult['OVIS_items_per_page']) ? $sResult['OVIS_items_per_page'] : 12;
				$sSettings['settings']['items_per_page_no_filter'] 			= isset($sResult['OVIS_items_per_page_no_filter']) ? $sResult['OVIS_items_per_page_no_filter'] : 12;
				$sSettings['settings']['items_per_page_no_filter_default'] 	= isset($sResult['OVIS_items_per_page_no_filter']) ? $sResult['OVIS_items_per_page_no_filter'] : 12;
				$sSettings['settings']['items_per_row'] 					= isset($sResult['OVIS_items_per_row']) ? $sResult['OVIS_items_per_row'] : 3;
				$sSettings['settings']['items_per_row_default'] 			= isset($sResult['OVIS_items_per_row']) ? $sResult['OVIS_items_per_row'] : 3;
				$sSettings['settings']['position_video_in_gallery'] 		= isset($sResult['OVIS_position_video_in_gallery']) ? $sResult['OVIS_position_video_in_gallery'] : 'second';
				$sSettings['settings']['bg_color'] 							= isset($sResult['OVIS_bg_color']) ? $sResult['OVIS_bg_color'] : '#f6f6f6';
				$sSettings['settings']['text_color'] 						= isset($sResult['OVIS_text_color']) ? $sResult['OVIS_text_color'] : '#333333';
				$sSettings['settings']['sessionclean'] 						= isset($sResult['OVIS_sessionclean']) ? $sResult['OVIS_sessionclean'] : false;
				$sSettings['settings']['custom_css'] 						= isset($sResult['OVIS_custom_css']) ? unserialize(base64_decode($sResult['OVIS_custom_css'])) : false;
				$sSettings['settings']['texts'] 							= isset($sResult['OVIS_texts']) ? unserialize(base64_decode($sResult['OVIS_texts'])) : false;
				$sSettings['settings']['rental_button_default'] 			= array('nl'=>'Verhuuraanvraag','en'=>'Rental request','de'=>'Mietanfrage','fr'=>'Demande de location');
				$sSettings['settings']['rental_opmerking_default'] 			= array('nl'=>'Omschrijving','en'=>'Description','de'=>'Beschreibung','fr'=>'Description');
				$sSettings['settings']['koopknop_default'] 					= array('nl'=>'Direct kopen','en'=>'Buy now','de'=>'Kaufe jetzt','fr'=>'Acheter maintenant');
				$sSettings['settings']['anchor'] 							= isset($sResult['OVIS_anchor']) ? $sResult['OVIS_anchor'] : false;
			}
		}
		
		return $sSettings;
	}

	
	/**
	 * Save settings
	 *
	 * @param [type] $settings
	 * @return bool Returns true on success
	 */
	public function save_settings($settings) {
		
		global $wpdb;

		if(!empty($settings['items_per_page'])) {
			if((int)$settings['items_per_page'] < 1) {
				$settings['items_per_page'] = 1;
			} elseif((int)$settings['items_per_page'] > 100) {
				$settings['items_per_page'] = 100;
			}
		} else {
			$settings['items_per_page'] = 12;
		}
		
		if(!empty($settings['items_per_page_no_filter'])) {
			if((int)$settings['items_per_page_no_filter'] < 1) {
				$settings['items_per_page_no_filter'] = 1;
			} elseif((int)$settings['items_per_page_no_filter'] > 100) {
				$settings['items_per_page_no_filter'] = 100;
			}
		} else {
			$settings['items_per_page_no_filter'] = 12;
		}
		
		if(!empty($settings['items_per_row'])) {
			if((int)$settings['items_per_row'] < 2) {
				$settings['items_per_row'] = 2;
			} elseif((int)$settings['items_per_row'] > 5) {
				$settings['items_per_row'] = 5;
			}
		} else {
			$settings['items_per_row'] = 3;
		}

		if(empty($settings['position_video_in_gallery'])) {
			$settings['position_video_in_gallery'] = 'second';
		}

		$sAuthKeyRaw = $settings['authentication_key'];
		set_transient('ovis_auth_hash', $sAuthKeyRaw, 30 * 24 * 60 * 60); // 30 days
		$authenticationKeyFile = __DIR__ . '/../include/.ovis_auth.secret';
		$sAuthKeyHash = hash('sha256', trim($sAuthKeyRaw));
		file_put_contents($authenticationKeyFile, $sAuthKeyHash);

		$wpdb->update(
			$wpdb->prefix . 'ovis',
			array( 
				'OVIS_authentication' => $settings['authentication_key'],
				'OVIS_button_bg_color' => $settings['button_bg_color'],
				'OVIS_button_text_color' => $settings['button_text_color'],
				'OVIS_object_bg_color' => $settings['object_bg_color'],
				'OVIS_object_text_color' => $settings['object_text_color'],
				'OVIS_bg_color' => $settings['bg_color'],
				'OVIS_text_color' => $settings['text_color'],
				'OVIS_items_per_page' => $settings['items_per_page'],
				'OVIS_items_per_page_no_filter' => $settings['items_per_page_no_filter'],
				'OVIS_items_per_row' => $settings['items_per_row'],
				'OVIS_position_video_in_gallery' => $settings['position_video_in_gallery']
			),
			array(
				'OVIS_ID' =>'1'
			)
		);
		
		return true;
	}


	/**
	 * Save appearance settings
	 *
	 * @param [type] $settings
	 * @return bool Returns true on success
	 */
	public function save_appearance($settings) {
	
		global $wpdb;
		
		$wpdb->update(
			$wpdb->prefix . 'ovis',
			array( 
				'OVIS_custom_css' => $settings['ovis_custom_css'],
				'OVIS_texts' => $settings['ovis_texts'],
				'OVIS_anchor' => $settings['ovis_anchor']
			),
			array(
				'OVIS_ID' =>'1'
			)
		);
		
		return true;
	}


	/**
	 * Clear OVIS cache
	 *
	 * @return bool Returns true on success
	 */
	public function clear_cache() {
		
		$bReturn = false;
		
		/**
		 * Retrieve & store the OVIS cache directory path
		 */
		$sImageURL = ABSPATH . 'wp-content/cache/ovis/';

		/**
		 * Check for directory
		 */
		if(is_dir($sImageURL)) {

			/**
			 * Retrieve & store current time
			 * 
			 * @link https://php.net/manual/en/function.time.php
			 */
			$iCurrentTime = time();
			
			$bReturn = true;
			
			/**
			 * Loop through all files & directories in directory
			 */
			foreach(scandir($sImageURL) as $file) {
				
				/**
				 * Don't include ./ & ../ directories
				 */
				if($file !== '.' && $file !== '..') {
					
					/**
					 * Retrieve & store last modified time of file
					 */
					$iFileTime = filemtime($sImageURL . $file);

					/**
					 * Check for file
					 */
					if(is_file($sImageURL . $file)) {

						/**
						 * Check if file is older then 2 hours
						 */
						if(($iCurrentTime - $iFileTime) >= (60 * 60 * 2)) {

							/**
							 * Delete file
							 */
							unlink($sImageURL . $file);
						}
					}
				}
			}
		}
		
		return $bReturn;
	}

}