<?php

/**
 * Utility class
 */
class Utility {

    /**
     * Generate CSRF token
     * 
     * @param string $sElementId
     * @return void
     */
    public static function generateCsrfToken($sElementId = 'csrf_token') {
        $sCsrfToken = bin2hex(random_bytes(32));
        $_SESSION[$sElementId] = $sCsrfToken;
    }


    /**
     * Helper function to check if object is empty
     * 
     * @param object $object
     * @return bool
     */
    public static function isObjectEmpty($object) {
        
        /**
         * Cast object to array then count array
         */
        if(count((array)$object) == 0) {

            /**
             * Object is empty
             */
            return true;
        }

        /**
         * Object is not empty
         */
        return false;
    }


    /**
     * Normalize given argument to boolean
     *
     * @param string|bool $stringOrBoolValue
     * 
     * @return bool
     */
    public static function normalizeBool($stringOrBoolValue) {

        /**
         * Check for valid given type
         */
        if( !(is_string($stringOrBoolValue)) && !(is_bool($stringOrBoolValue)) ) {
            return false;
        }

        /**
         * Check if parameter is boolean
         */
        if( is_bool($stringOrBoolValue) ) {
            return $stringOrBoolValue;
        }

        /**
         * Check for empty value
         */
        if( empty($stringOrBoolValue) ) {
            return false;
        }

        $stringOrBoolValue = trim(strtolower($stringOrBoolValue));

        if( $stringOrBoolValue === 'yes' || $stringOrBoolValue === 'ja'|| $stringOrBoolValue === 'y' ) {
            return true;
        }
        
        return false;
    }


    /**
     * Normalize given argument to array
     * 
     * @param string|array $stringOrArrayValue
     * 
     * @return array
     */
    public static function normalizeArray($stringOrArrayValue) {

        /**
         * Check for valid given type
         */
        if( !(is_string($stringOrArrayValue)) && !(is_array($stringOrArrayValue)) ) {
            return array();
        }

        /**
         * Check if parameter is array
         */
        if( is_array($stringOrArrayValue) ) {
            return $stringOrArrayValue;
        }

        /**
         * Check for empty value
         */
        if( empty($stringOrArrayValue) ) {
            return array();
        }
        
        return explode(',', $stringOrArrayValue);
    }


    /**
     * Check given options against list of available options
     *
     * @param string|array $options
     * @param array $aAvailableOptions
     * @return array
     */
    public static function validateAndNormalizeOptions($options, $aAvailableOptions) {
        
        $aValidatedOptions = array();

        /**
         * Flatten the available options into a single array of valid values
         */
        $aValidAvailableOptions = array_unique(
            array_merge(
                array_values($aAvailableOptions), 
                array_keys($aAvailableOptions)
            )
        );

        /**
         * Ensure given parameter is an array
         */
        if( !is_array($options) ) {
            $options = array_map('trim', explode(',', ($options)));
        }

        /**
         * Loop through all options
         */
        foreach( $options as $option ) {

            /**
             * Check if current option is a valid available option
             */
            if( in_array($option, $aValidAvailableOptions, true) ) {

                /**
                 * Check if it's a Dutch option
                 */
                if( isset($aAvailableOptions[$option]) ) {

                    /**
                     * Convert it to the official option name
                     */
                    array_push( $aValidatedOptions, $aAvailableOptions[$option] );
                } else {

                    /**
                     * If it's already the official option
                     */
                    array_push( $aValidatedOptions, $option );
                }
            }
        }

        return $aValidatedOptions;
    }


    /**
     * Map backend error codes to frontend field names.
     *
     * @param array $aErrors - List of backend error codes
     * @param array $aErrorMapping - Mapping array (backend => frontend)
     * @return array List of mapped frontend fields
     */
    public static function mapErrors($aErrors, $aErrorMapping) {
        $aMapped = array();

        foreach($aErrors as $err) {
            if(isset($aErrorMapping[$err])) {
                $mappedField = $aErrorMapping[$err];
                if(is_array($mappedField)) {
                    $aMapped = array_merge($aMapped, $mappedField);
                } else {
                    array_push($aMapped, $mappedField);
                }
            }
        }
    
        return $aMapped;
    }
}